package awaybuilder
{
	import away3d.containers.View3D;
	
	import awaybuilder.abstracts.AbstractBuilder;
	import awaybuilder.abstracts.AbstractCameraController;
	import awaybuilder.abstracts.AbstractGeometryController;
	import awaybuilder.abstracts.AbstractParser;
	import awaybuilder.camera.AnimationControl;
	import awaybuilder.camera.CameraController;
	import awaybuilder.camera.CameraFocus;
	import awaybuilder.camera.CameraZoom;
	import awaybuilder.collada.ColladaParser;
	import awaybuilder.events.CameraEvent;
	import awaybuilder.events.GeometryEvent;
	import awaybuilder.events.SceneEvent;
	import awaybuilder.geometry.GeometryController;
	import awaybuilder.interfaces.IAssetContainer;
	import awaybuilder.interfaces.ICameraController;
	import awaybuilder.interfaces.ISceneContainer;
	import awaybuilder.parsers.SceneXMLParser;
	import awaybuilder.utils.CoordinateCopy;
	import awaybuilder.vo.SceneCameraVO;
	import awaybuilder.vo.SceneGeometryVO;
	
	import flash.display.BitmapData;
	import flash.display.DisplayObject;
	import flash.display.Sprite;
	import flash.events.Event;
	
	
	
	public class WorldBuilder extends Sprite implements IAssetContainer , ISceneContainer , ICameraController
	{
		public var source : String = SceneSource.MAYA ;
		public var data : * ;
		public var precision : uint = ScenePrecision.PERFECT ;
		public var startCamera : String ;
		public var update : String = SceneUpdate.CONTINUOUS ;
		public var cameraZoom : Number = CameraZoom.MAYA_W720_H502 ;
		public var animationControl : String = AnimationControl.INTERNAL ;

		protected var view : View3D ;
		protected var parser : AbstractParser ;
		protected var builder : AbstractBuilder ;
		protected var cameraController : AbstractCameraController ;
		protected var geometryController : AbstractGeometryController ;

		
		
		public function WorldBuilder ( )
		{
			this.builder = new SceneBuilder ( ) ;
			this.addEventListener ( Event.ADDED_TO_STAGE , this.onAddedToDisplayList ) ;
		}
		
		
		
		////////////////////////////////////////////////////////////////////////////////
		//
		// Public Methods
		//
		////////////////////////////////////////////////////////////////////////////////
		
		
		
		public function build ( ) : void
		{
			this.createView ( ) ;
			this.createParser ( ) ;
		}
		
		
		
		public function addBitmapDataAsset ( id : String , data : BitmapData ) : void
		{
			this.builder.addBitmapDataAsset ( id , data ) ;
		}

		
		
		public function addDisplayObjectAsset ( id : String , data : DisplayObject ) : void
		{
			this.builder.addDisplayObjectAsset ( id , data ) ;
		}

		
		
		public function addColladaAsset ( id : String , data : XML ) : void
		{
			this.builder.addColladaAsset ( id , data ) ;
		}
		
		
		
		public function getCameras ( ) : Array
		{
			return this.builder.getCameras ( ) ;
		}

		
		
		public function getGeometry ( ) : Array
		{
			return this.builder.getGeometry ( ) ;
		}

		
		
		public function getSections ( ) : Array
		{
			return this.builder.getSections ( ) ;
		}

		
		
		public function getCameraById ( id : String ) : SceneCameraVO
		{
			return this.builder.getCameraById ( id ) ;
		}

		
		
		public function getGeometryById ( id : String ) : SceneGeometryVO
		{
			return this.builder.getGeometryById ( id ) ;
		}
		
		
		
		public function navigateTo ( vo : SceneCameraVO ) : void
		{
			this.cameraController.navigateTo ( vo ) ;
		}
		
		
		
		public function teleportTo ( vo : SceneCameraVO ) : void
		{
			this.cameraController.teleportTo ( vo ) ;
		}

		
		
		////////////////////////////////////////////////////////////////////////////////
		//
		// Protected Methods
		//
		////////////////////////////////////////////////////////////////////////////////
		
		
		
		protected function createView ( ) : void
		{
			this.view = new View3D ( ) ;
			this.addChild ( this.view ) ;
		}
		
		
		
		protected function createParser ( ) : void
		{
			switch ( this.source )
			{
				case SceneSource.MAYA :
				{
					this.parser = new ColladaParser ( ) ;
					break ;
				}
				case SceneSource.NATIVE :
				{
					this.parser = new SceneXMLParser ( ) ;
					break ;
				}
			}
			
			this.parser.addEventListener ( Event.COMPLETE , this.onParsingComplete ) ;
			this.parser.parse ( this.data ) ;
		}
		
		
		
		protected function setupCamera ( ) : void
		{
			switch ( this.source )
			{
				case SceneSource.MAYA :
				{
					this.view.camera.focus = CameraFocus.MAYA ;
					this.cameraZoom ? this.view.camera.zoom = CameraZoom.MAYA_W720_H502 : this.view.camera.zoom = this.cameraZoom ;
					break ;
				}
				case SceneSource.NATIVE :
				{
					this.view.camera.focus = CameraFocus.DEFAULT ;
					this.cameraZoom ? this.view.camera.zoom = CameraZoom.DEFAULT : this.view.camera.zoom = this.cameraZoom ;
					break ;
				}
			}
			
			if ( this.startCamera )
			{
				var vo : SceneCameraVO = this.builder.getCameraById ( this.startCamera ) ;
				
				CoordinateCopy.position ( vo.camera , this.view.camera ) ;
				CoordinateCopy.rotation ( vo.camera , this.view.camera ) ;
			}
		}
		
		
		
		protected function setupBuilder ( ) : void
		{
			var builder : SceneBuilder = this.builder as SceneBuilder ;
			var sections : Array = this.parser.sections ;
			
			switch ( this.source )
			{
				case SceneSource.MAYA :
				{
					builder.coordinateSystem = CoordinateSystem.MAYA ;
					break ;
				}
				case SceneSource.NATIVE :
				{
					builder.coordinateSystem = CoordinateSystem.NATIVE ;
					break ;
				}
			}
			
			builder.precision = this.precision ;
			builder.addEventListener ( SceneEvent.RENDER , this.render ) ;
			builder.addEventListener ( Event.COMPLETE , this.onBuildingComplete ) ;
			builder.build ( this.view , sections ) ;
		}
		
		
		
		protected function createCameraController ( ) : void
		{
			this.cameraController = new CameraController ( this.view.camera ) ;
			this.cameraController.update = this.update ;
			this.cameraController.animationControl = this.animationControl ;
			
			if ( this.startCamera )
			{
				var startCamera : SceneCameraVO = this.builder.getCameraById ( this.startCamera ) ;
				
				this.cameraController.startCamera = startCamera ;
				this.cameraController.teleportTo ( startCamera ) ;
			}
			
			this.cameraController.addEventListener ( SceneEvent.RENDER , this.render ) ;
			this.cameraController.addEventListener ( CameraEvent.ANIMATION_START , this.onCameraEvent ) ;
			this.cameraController.addEventListener ( CameraEvent.ANIMATION_COMPLETE , this.onCameraEvent ) ;
		}
		
		
		
		protected function createGeometryController ( ) : void
		{
			this.geometryController = new GeometryController ( this.builder.getGeometry ( ) ) ;
			this.geometryController.addEventListener ( GeometryEvent.DOWN , this.onGeometryEvent ) ;
			this.geometryController.addEventListener ( GeometryEvent.MOVE , this.onGeometryEvent ) ;
			this.geometryController.addEventListener ( GeometryEvent.OUT , this.onGeometryEvent ) ;
			this.geometryController.addEventListener ( GeometryEvent.OVER , this.onGeometryEvent ) ;
			this.geometryController.addEventListener ( GeometryEvent.UP , this.onGeometryEvent ) ;
			this.geometryController.enableInteraction ( ) ;
		}
		
		
		
		protected function setupSceneUpdate ( ) : void
		{
			switch ( this.update )
			{
				/* NOTE: Render events dispatched by the camera controller.
				case SceneUpdate.ON_CAMERA_UPDATE :
				*/
				case SceneUpdate.CONTINUOUS :
				{
					this.addEventListener ( Event.ENTER_FRAME , this.render ) ;
					break ;
				}
			}
		}
		
		
		
		protected function render ( event : Event = null ) : void
		{
			switch ( this.update )
			{
				/* NOTE: Render events handled externally.
				case SceneUpdate.MANUAL :
				*/
				case SceneUpdate.CONTINUOUS :
				case SceneUpdate.ON_CAMERA_UPDATE :
				{
					this.view.render ( ) ;
					break ;
				}
			}
			
			this.dispatchEvent ( new SceneEvent ( SceneEvent.RENDER ) ) ;
		}
		
		
		
		////////////////////////////////////////////////////////////////////////////////
		//
		// Event Handlers
		//
		////////////////////////////////////////////////////////////////////////////////
		
		
		
		protected function onAddedToDisplayList ( event : Event ) : void
		{
			this.removeEventListener ( Event.ADDED_TO_STAGE , this.onAddedToDisplayList ) ;
			this.visible = false ;
		}
		
		
		
		protected function onParsingComplete ( event : Event ) : void
		{
			this.setupBuilder ( ) ;
		}
		
		
		
		protected function onBuildingComplete ( event : Event ) : void
		{
			this.setupCamera ( ) ;
			this.createCameraController ( ) ;
			this.createGeometryController ( ) ;
			this.setupSceneUpdate ( ) ;
			this.render ( ) ;
			
			this.visible = true ;
			this.dispatchEvent ( new Event ( Event.COMPLETE ) ) ;
		}

		
		
		protected function onGeometryEvent ( event : GeometryEvent ) : void
		{
			var type : String = event.type ;
			var geometry : SceneGeometryVO = event.geometry ;
			var geometryEvent : GeometryEvent = new GeometryEvent ( type ) ;
			
			switch ( type )
			{
				case GeometryEvent.UP :
				{
					if ( geometry.targetCamera )
					{
						var camera : SceneCameraVO = this.builder.getCameraById ( geometry.targetCamera ) ;
						this.cameraController.navigateTo ( camera ) ;
					}
					break ;
				}
			}
			
			geometryEvent.geometry = geometry ;
			this.dispatchEvent ( geometryEvent ) ;
		}
		
		
		
		protected function onCameraEvent ( event : CameraEvent ) : void
		{
			var cameraEvent : CameraEvent = new CameraEvent ( event.type ) ;
			
			cameraEvent.targetCamera = event.targetCamera ;
			this.dispatchEvent ( cameraEvent ) ;
		}
		
		
		
		////////////////////////////////////////////////////////////////////////////////
		//
		// Getters and Setters
		//
		////////////////////////////////////////////////////////////////////////////////
		
		
		
		public function get worldView ( ) : View3D
		{
			return this.view ;
		}
	}
}